-- Ascension: Modifiers to increase difficulty

local biomes = require "defs.biomes"
local kassert = require "util.kassert"
local lume = require "util.lume"


-- See also AscensionModifierSource in tuning.lua.
local ascensions = {
	-- Elite enemies spawn
	{
		stringkey = STRINGS.ASCENSIONS.ASCENSION_ONE,
		icon = "images/map_ftf/frenzy_2.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- Enemies are more aggressive
	{
		stringkey = STRINGS.ASCENSIONS.ASCENSION_TWO,
		icon = "images/map_ftf/frenzy_3.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- Enemies have more health
	{
		stringkey = STRINGS.ASCENSIONS.ASCENSION_THREE,
		icon = "images/map_ftf/frenzy_4.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- More enemies spawn
	{
		stringkey = STRINGS.ASCENSIONS.MORE_ENEMIES,
		icon = "images/map_ftf/frenzy_5.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- Bosses have more health
	{
		stringkey = STRINGS.ASCENSIONS.BOSS_HEALTH_BOOST,
		icon = "images/map_ftf/frenzy_6.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- More elite enemies spawn
	{
		stringkey = STRINGS.ASCENSIONS.MORE_ELITES,
		icon = "images/map_ftf/frenzy_7.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- Elites are more aggressive
	{
		stringkey = STRINGS.ASCENSIONS.AGGRESSIVE_ELITES,
		icon = "images/map_ftf/frenzy_8.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	-- Bosses are more aggressive
	{
		stringkey = STRINGS.ASCENSIONS.AGGRESSIVE_BOSSES,
		icon = "images/map_ftf/frenzy_9.tex",
		func = function(inst)
			-- For creature tuning, see AscensionModifierSource in tuning.lua.
		end,
	},

	--]]
}

local AscensionManager = Class(function(self, inst)
	self.inst = inst -- This is TheDungeon.progression
	self.current = 0
	self.ascension_data = ascensions
	self.num_ascensions = #ascensions
	self.persistdata = {
		last_selected = {},
	}
end)

function AscensionManager:OnSave()
	local data = {}
	data.last_selected = self.persistdata.last_selected
	return data
end

function AscensionManager:OnLoad(data)
	print("------------------- AscensionManager:OnLoad(data)")
	-- If we don't deepcopy, then we'll write directly to TheSaveSystem which
	-- might be unpredictable!
	self.persistdata.last_selected = deepcopy(data.last_selected)

	-- Load the current level.
	self:_SetCurrentLevel(self:_GetDesiredLevel())
end

function AscensionManager:OnPostLoadWorld()
	-- I think this is the earliest callback after loading save data to apply
	-- ascension. If you need ascension already applied, see OnStartRoom
	-- callback or check GetCurrentLevel in your init (it will always be
	-- correct for the host because it's set before world creation).
	self:_ActivateAscension(self:_GetDesiredLevel())
end

function AscensionManager:_GetDesiredLevel()
	local biome_location = TheDungeon:GetDungeonMap():GetBiomeLocation()
	if TheNet:IsHost() then
		return self:GetSelectedAscension(biome_location.id)
	else
		local _mode, _seqNr, dungeon_run_params, _quest_params = TheNet:GetRunData() -- client
		return dungeon_run_params.ascension or 0
	end
end

function AscensionManager:Debug_SetAscension(level, location_id)
	location_id = location_id or TheDungeon:GetDungeonMap():GetBiomeLocation().id
	level = self:_ActivateAscension(level)
	self:StoreSelectedAscension(location_id, self.current)
	self.inst:WriteProgression()
	TheSaveSystem:SaveAll() -- Non debug use should use callback!
end

function AscensionManager:_SetCurrentLevel(level)
	level = math.min(level, self.num_ascensions)
	self.current = level
	return level
end

-- Applies the ascension level to the current world. Tuning is applied to
-- creatures when they spawn by querying the active ascension.
--
-- Use Debug_SetAscension for debug!
function AscensionManager:_ActivateAscension(level)
	-- Don't store the level because it might have come from the host.
	level = self:_SetCurrentLevel(level)
	--print("_ActivateAscension:", level)
	for i, asc in ipairs(ascensions) do
		--print(i)
		if i > level then
			break
		end
		--print(asc.stringkey)
		asc.func(TheWorld)
	end
end

-- Allow one higher than the highest completed for that location.
function AscensionManager:GetMaxAllowedLevelForParty(location)
	local best = self:GetHighestCompletedLevelForParty(location.id) or -1
	local allowed = lume.clamp(best + 1, 0, self.num_ascensions)

	-- TEMP cap max frenzy level at the last normal frenzy level
	allowed = math.min(allowed, NORMAL_FRENZY_LEVELS)

	return allowed
end

function AscensionManager:GetCurrentLevel()
	return self.current
end

function AscensionManager:GetSelectedAscension(location_id)
	kassert.typeof("string", location_id)

	printf("------------------- AscensionManager:GetSelectedAscension(%s) [%s]", location_id, self.persistdata.last_selected[location_id])

	return self.persistdata.last_selected[location_id] or 0
end

-- Prefer Debug_SetAscension for debug use!
function AscensionManager:StoreSelectedAscension(location_id, value)
	kassert.typeof("string", location_id)

	printf("------------------- AscensionManager:StoreSelectedAscension(%s, %d)", location_id, value)

	self.persistdata.last_selected[location_id] = value
end

-- The highest ascension that any party member has unlocked.
function AscensionManager:GetHighestCompletedLevelForParty(location_id)
	kassert.typeof("string", location_id)

	local highest_unlock = nil

	for _, player in ipairs(AllPlayers) do
		local weapon = player.components.inventory:GetEquippedWeaponType()
		if weapon then 
			local level = player.components.unlocktracker:GetCompletedAscensionLevel(location_id, weapon) or -1
			if not highest_unlock or level > highest_unlock then
				highest_unlock = level
			end
		end
	end

	return highest_unlock
end

function AscensionManager:GetPartyAscensionData(location_id)
	kassert.typeof("string", location_id)

	local data = {}

	for _, player in ipairs(AllPlayers) do
		local playerID = player.Network:GetPlayerID()
		local weapon_type = player.components.inventory:GetEquippedWeaponType()
		local level = player.components.unlocktracker:GetCompletedAscensionLevel(location_id, weapon_type) or -1
		data[playerID] = { player = player, weapon_type = weapon_type, level = level }
	end

	return data
end

function AscensionManager:SetHighestCompletedAscension(player, location_id, weapon_type, num)
	player.components.unlocktracker:SetAscensionLevelCompleted(location_id, weapon_type, num)
end


function AscensionManager:CompleteCurrentAscension()
	local location_id = TheDungeon:GetDungeonMap().data.location_id

	for _, player in ipairs(AllPlayers) do
		local weapon_type = player.components.inventory:GetEquippedWeaponType()
		local level = player.components.unlocktracker:GetCompletedAscensionLevel(location_id, weapon_type)

		if self.current > level then
			-- players should only ever unlock 1 level at a time, even if the ascension they just completed is higher than that
			self:SetHighestCompletedAscension(player, location_id, weapon_type, level + 1)
		end
	end
end

function AscensionManager:DEBUG_UnlockAscension(location_id, num)
	num = num or #ascensions

	for _, player in ipairs(AllPlayers) do
		for _, weapon_type in pairs(WEAPON_TYPES) do
			if player.components.unlocktracker:IsWeaponTypeUnlocked(weapon_type) then
				self:SetHighestCompletedAscension(player, location_id, weapon_type, num)
			end
		end
	end

	self.persistdata.last_selected[location_id] = num
end

local dbg_location_id
function AscensionManager:DebugDrawEntity(ui, panel, colors)
	local location
	dbg_location_id, location = biomes._BiomeLocationPicker(ui, dbg_location_id)

	local highest = self:GetHighestCompletedLevelForParty(dbg_location_id)
	ui:Value("GetCurrentLevel", self:GetCurrentLevel())
	ui:Value("GetHighestCompletedLevelForParty", highest)
	ui:Value("GetMaxAllowedLevelForParty", self:GetMaxAllowedLevelForParty(location))
	ui:Value("GetSelectedAscension", self:GetSelectedAscension(dbg_location_id))
	panel:AppendTable(ui, self:GetPartyAscensionData(dbg_location_id), "GetPartyAscensionData")
	--~ self:GetLootEligibility(GetDebugPlayer(), dbg_location_id, weapon_type, level)
end

function AscensionManager:GetLootEligibility(player, location_id, weapon_type, level)
	local highest = player.components.unlocktracker:GetCompletedAscensionLevel(location_id, weapon_type)
	if not highest then return true end -- this player has never done ascension with this weapon/ location combo so they must be eligible
	return highest < level
end

function AscensionManager:IsEligibleForHeart(player)
	local location_id = TheDungeon:GetDungeonMap().data.location_id
	local weapon_type = player.components.inventory:GetEquippedWeaponType()
	return weapon_type
		and self:GetLootEligibility(player, location_id, weapon_type, self:GetCurrentLevel())
		or false
end

function AscensionManager.GetActiveModsPerLevel(frenzy_level)
	local mods_per_level = {
		[1] = {
	        [FRENZY_MODIFIERS.id.STRONG_ENEMIES] = 1,
	        [FRENZY_MODIFIERS.id.ELITE_SPAWN] = 1,
	        [FRENZY_MODIFIERS.id.MINIBOSS_EQUIPMENT] = 1,
	        [FRENZY_MODIFIERS.id.MORE_LOOT] = 1,
	    },
	    [2] = {
	        [FRENZY_MODIFIERS.id.STRONG_ENEMIES] = 2,
	        [FRENZY_MODIFIERS.id.AGGRESSIVE_ENEMIES] = 1,
	        [FRENZY_MODIFIERS.id.MINIBOSS_EQUIPMENT] = 1,
	        [FRENZY_MODIFIERS.id.ELITE_SPAWN] = 2,
	        [FRENZY_MODIFIERS.id.MORE_LOOT] = 2,
	        [FRENZY_MODIFIERS.id.REVIVE_COST] = 1,
	        [FRENZY_MODIFIERS.id.INGOTS_DROP] = 1,
	    },
	    [3] = {
	        [FRENZY_MODIFIERS.id.STRONG_ENEMIES] = 3,
	        [FRENZY_MODIFIERS.id.AGGRESSIVE_ENEMIES] = 2,
	        [FRENZY_MODIFIERS.id.MINIBOSS_EQUIPMENT] = 1,
	        [FRENZY_MODIFIERS.id.ELITE_SPAWN] = 2,
	        [FRENZY_MODIFIERS.id.MINIBOSS_SPAWN] = 1,
	        [FRENZY_MODIFIERS.id.MORE_LOOT] = 3,
	        [FRENZY_MODIFIERS.id.REVIVE_COST] = 1,
	        [FRENZY_MODIFIERS.id.INGOTS_DROP] = 1,
	    },
	}
	return mods_per_level[frenzy_level]
end

function AscensionManager.GetModDescription(frenzy_modifier, mod_level)
	local descriptions = {
		[FRENZY_MODIFIERS.id.STRONG_ENEMIES] = {
	        [1] = STRINGS.ASCENSIONS.STRONG_ENEMIES_1,
	        [2] = STRINGS.ASCENSIONS.STRONG_ENEMIES_2,
	        [3] = STRINGS.ASCENSIONS.STRONG_ENEMIES_3,
	    },
	    [FRENZY_MODIFIERS.id.AGGRESSIVE_ENEMIES] = {
	        [1] = STRINGS.ASCENSIONS.AGGRESSIVE_ENEMIES_1,
	        [2] = STRINGS.ASCENSIONS.AGGRESSIVE_ENEMIES_2,
	    },
	    [FRENZY_MODIFIERS.id.MORE_LOOT] = {
	        [1] = STRINGS.ASCENSIONS.MORE_LOOT_1,
	        [2] = STRINGS.ASCENSIONS.MORE_LOOT_2,
	        [3] = STRINGS.ASCENSIONS.MORE_LOOT_3,
	    },
	    [FRENZY_MODIFIERS.id.ELITE_SPAWN] = {
	        [1] = STRINGS.ASCENSIONS.ELITE_SPAWN_1,
	        [2] = STRINGS.ASCENSIONS.ELITE_SPAWN_2,
	    },
	    [FRENZY_MODIFIERS.id.MINIBOSS_SPAWN] = {
	        [1] = STRINGS.ASCENSIONS.MINIBOSS_SPAWN_1,
	    },
	    [FRENZY_MODIFIERS.id.MINIBOSS_EQUIPMENT] = {
	        [1] = STRINGS.ASCENSIONS.MINIBOSS_EQUIPMENT_1,
	    },
	    [FRENZY_MODIFIERS.id.REVIVE_COST] = {
	        [1] = STRINGS.ASCENSIONS.REVIVE_COST_1,
	    },
	    [FRENZY_MODIFIERS.id.INGOTS_DROP] = {
	        [1] = STRINGS.ASCENSIONS.INGOTS_DROP_1,
	    },
	}
	return descriptions[frenzy_modifier][mod_level or 1]
end

return AscensionManager

--[[
Ideas:
Increase Difficulty 3 room spawn rate
Increase small enemy damage, Decrease enemy cooldown time
Increase big enemy damage, Decrease big enemy cooldown time
Increase boss damage, Decrease boss cooldown time
Potion heals for 75%
Start with 90% health
Increase normal enemy health by 2x
Increase big enemy health by 2x (Yammo)
Increase boss health by 2x
10% less max HP
Everything costs more.
Double boss.
Traps do extra damage to players
WaitForDefeatedCount and WaitForDefeatedPercentage modifier

--]]
